/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

/**
 * In bug 494140 we found out that creating an exception to a series duplicates
 * alarms. This unit test makes sure the alarms don't duplicate themselves. The
 * same goes for relations and attachments.
 */
add_task(async () => {
  const storageCal = getStorageCal();

  const item = createEventFromIcalString(
    "BEGIN:VEVENT\r\n" +
      "CREATED:20090603T171401Z\r\n" +
      "LAST-MODIFIED:20090617T080410Z\r\n" +
      "DTSTAMP:20090617T080410Z\r\n" +
      "UID:c1a6cfe7-7fbb-4bfb-a00d-861e07c649a5\r\n" +
      "SUMMARY:Test\r\n" +
      "DTSTART:20090603T073000Z\r\n" +
      "DTEND:20090603T091500Z\r\n" +
      "RRULE:FREQ=DAILY;COUNT=5\r\n" +
      "RELATED-TO:RELTYPE=SIBLING:<foo@example.org>\r\n" +
      "ATTACH:http://www.example.org/\r\n" +
      "BEGIN:VALARM\r\n" +
      "ACTION:DISPLAY\r\n" +
      "TRIGGER;VALUE=DURATION:-PT10M\r\n" +
      "DESCRIPTION:Mozilla Alarm: Test\r\n" +
      "END:VALARM\r\n" +
      "END:VEVENT"
  );

  // There should be one alarm, one relation and one attachment
  equal(item.getAlarms().length, 1);
  equal(item.getRelations().length, 1);
  equal(item.getAttachments().length, 1);

  // Change the occurrence to another day
  const occ = item.recurrenceInfo.getOccurrenceFor(cal.createDateTime("20090604T073000Z"));
  occ.QueryInterface(Ci.calIEvent);
  occ.startDate = cal.createDateTime("20090618T073000Z");
  item.recurrenceInfo.modifyException(occ, true);

  // There should still be one alarm, one relation and one attachment
  equal(item.getAlarms().length, 1);
  equal(item.getRelations().length, 1);
  equal(item.getAttachments().length, 1);

  // Add the item to the storage calendar and retrieve it again
  await storageCal.adoptItem(item);

  const retrievedItem = await storageCal.getItem("c1a6cfe7-7fbb-4bfb-a00d-861e07c649a5");
  // There should still be one alarm, one relation and one attachment
  equal(retrievedItem.getAlarms().length, 1);
  equal(retrievedItem.getRelations().length, 1);
  equal(retrievedItem.getAttachments().length, 1);
});
