/*
 * QEMU model of the Clock-Reset-LPD (CRL).
 *
 * Copyright (c) 2022 Xilinx Inc.
 * Copyright (c) 2025 Advanced Micro Devices, Inc.
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * Written by Edgar E. Iglesias <edgar.iglesias@xilinx.com>
 */
#ifndef HW_MISC_XLNX_VERSAL_CRL_H
#define HW_MISC_XLNX_VERSAL_CRL_H

#include "hw/sysbus.h"
#include "hw/register.h"
#include "target/arm/cpu-qom.h"
#include "hw/arm/xlnx-versal-version.h"

#define TYPE_XLNX_VERSAL_CRL_BASE "xlnx-versal-crl-base"
#define TYPE_XLNX_VERSAL_CRL "xlnx-versal-crl"
#define TYPE_XLNX_VERSAL2_CRL "xlnx-versal2-crl"

OBJECT_DECLARE_TYPE(XlnxVersalCRLBase, XlnxVersalCRLBaseClass,
                    XLNX_VERSAL_CRL_BASE)
OBJECT_DECLARE_SIMPLE_TYPE(XlnxVersalCRL, XLNX_VERSAL_CRL)
OBJECT_DECLARE_SIMPLE_TYPE(XlnxVersal2CRL, XLNX_VERSAL2_CRL)

REG32(ERR_CTRL, 0x0)
    FIELD(ERR_CTRL, SLVERR_ENABLE, 0, 1)
REG32(IR_STATUS, 0x4)
    FIELD(IR_STATUS, ADDR_DECODE_ERR, 0, 1)
REG32(IR_MASK, 0x8)
    FIELD(IR_MASK, ADDR_DECODE_ERR, 0, 1)
REG32(IR_ENABLE, 0xc)
    FIELD(IR_ENABLE, ADDR_DECODE_ERR, 0, 1)
REG32(IR_DISABLE, 0x10)
    FIELD(IR_DISABLE, ADDR_DECODE_ERR, 0, 1)
REG32(WPROT, 0x1c)
    FIELD(WPROT, ACTIVE, 0, 1)
REG32(PLL_CLK_OTHER_DMN, 0x20)
    FIELD(PLL_CLK_OTHER_DMN, APLL_BYPASS, 0, 1)
REG32(RPLL_CTRL, 0x40)
    FIELD(RPLL_CTRL, POST_SRC, 24, 3)
    FIELD(RPLL_CTRL, PRE_SRC, 20, 3)
    FIELD(RPLL_CTRL, CLKOUTDIV, 16, 2)
    FIELD(RPLL_CTRL, FBDIV, 8, 8)
    FIELD(RPLL_CTRL, BYPASS, 3, 1)
    FIELD(RPLL_CTRL, RESET, 0, 1)
REG32(RPLL_CFG, 0x44)
    FIELD(RPLL_CFG, LOCK_DLY, 25, 7)
    FIELD(RPLL_CFG, LOCK_CNT, 13, 10)
    FIELD(RPLL_CFG, LFHF, 10, 2)
    FIELD(RPLL_CFG, CP, 5, 4)
    FIELD(RPLL_CFG, RES, 0, 4)
REG32(RPLL_FRAC_CFG, 0x48)
    FIELD(RPLL_FRAC_CFG, ENABLED, 31, 1)
    FIELD(RPLL_FRAC_CFG, SEED, 22, 3)
    FIELD(RPLL_FRAC_CFG, ALGRTHM, 19, 1)
    FIELD(RPLL_FRAC_CFG, ORDER, 18, 1)
    FIELD(RPLL_FRAC_CFG, DATA, 0, 16)
REG32(PLL_STATUS, 0x50)
    FIELD(PLL_STATUS, RPLL_STABLE, 2, 1)
    FIELD(PLL_STATUS, RPLL_LOCK, 0, 1)
REG32(RPLL_TO_XPD_CTRL, 0x100)
    FIELD(RPLL_TO_XPD_CTRL, CLKACT, 25, 1)
    FIELD(RPLL_TO_XPD_CTRL, DIVISOR0, 8, 10)
REG32(LPD_TOP_SWITCH_CTRL, 0x104)
    FIELD(LPD_TOP_SWITCH_CTRL, CLKACT_ADMA, 26, 1)
    FIELD(LPD_TOP_SWITCH_CTRL, CLKACT, 25, 1)
    FIELD(LPD_TOP_SWITCH_CTRL, DIVISOR0, 8, 10)
    FIELD(LPD_TOP_SWITCH_CTRL, SRCSEL, 0, 3)
REG32(LPD_LSBUS_CTRL, 0x108)
    FIELD(LPD_LSBUS_CTRL, CLKACT, 25, 1)
    FIELD(LPD_LSBUS_CTRL, DIVISOR0, 8, 10)
    FIELD(LPD_LSBUS_CTRL, SRCSEL, 0, 3)
REG32(CPU_R5_CTRL, 0x10c)
    FIELD(CPU_R5_CTRL, CLKACT_OCM2, 28, 1)
    FIELD(CPU_R5_CTRL, CLKACT_OCM, 27, 1)
    FIELD(CPU_R5_CTRL, CLKACT_CORE, 26, 1)
    FIELD(CPU_R5_CTRL, CLKACT, 25, 1)
    FIELD(CPU_R5_CTRL, DIVISOR0, 8, 10)
    FIELD(CPU_R5_CTRL, SRCSEL, 0, 3)
REG32(IOU_SWITCH_CTRL, 0x114)
    FIELD(IOU_SWITCH_CTRL, CLKACT, 25, 1)
    FIELD(IOU_SWITCH_CTRL, DIVISOR0, 8, 10)
    FIELD(IOU_SWITCH_CTRL, SRCSEL, 0, 3)
REG32(GEM0_REF_CTRL, 0x118)
    FIELD(GEM0_REF_CTRL, CLKACT_RX, 27, 1)
    FIELD(GEM0_REF_CTRL, CLKACT_TX, 26, 1)
    FIELD(GEM0_REF_CTRL, CLKACT, 25, 1)
    FIELD(GEM0_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(GEM0_REF_CTRL, SRCSEL, 0, 3)
REG32(GEM1_REF_CTRL, 0x11c)
    FIELD(GEM1_REF_CTRL, CLKACT_RX, 27, 1)
    FIELD(GEM1_REF_CTRL, CLKACT_TX, 26, 1)
    FIELD(GEM1_REF_CTRL, CLKACT, 25, 1)
    FIELD(GEM1_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(GEM1_REF_CTRL, SRCSEL, 0, 3)
REG32(GEM_TSU_REF_CTRL, 0x120)
    FIELD(GEM_TSU_REF_CTRL, CLKACT, 25, 1)
    FIELD(GEM_TSU_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(GEM_TSU_REF_CTRL, SRCSEL, 0, 3)
REG32(USB0_BUS_REF_CTRL, 0x124)
    FIELD(USB0_BUS_REF_CTRL, CLKACT, 25, 1)
    FIELD(USB0_BUS_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(USB0_BUS_REF_CTRL, SRCSEL, 0, 3)
REG32(UART0_REF_CTRL, 0x128)
    FIELD(UART0_REF_CTRL, CLKACT, 25, 1)
    FIELD(UART0_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(UART0_REF_CTRL, SRCSEL, 0, 3)
REG32(UART1_REF_CTRL, 0x12c)
    FIELD(UART1_REF_CTRL, CLKACT, 25, 1)
    FIELD(UART1_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(UART1_REF_CTRL, SRCSEL, 0, 3)
REG32(SPI0_REF_CTRL, 0x130)
    FIELD(SPI0_REF_CTRL, CLKACT, 25, 1)
    FIELD(SPI0_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(SPI0_REF_CTRL, SRCSEL, 0, 3)
REG32(SPI1_REF_CTRL, 0x134)
    FIELD(SPI1_REF_CTRL, CLKACT, 25, 1)
    FIELD(SPI1_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(SPI1_REF_CTRL, SRCSEL, 0, 3)
REG32(CAN0_REF_CTRL, 0x138)
    FIELD(CAN0_REF_CTRL, CLKACT, 25, 1)
    FIELD(CAN0_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(CAN0_REF_CTRL, SRCSEL, 0, 3)
REG32(CAN1_REF_CTRL, 0x13c)
    FIELD(CAN1_REF_CTRL, CLKACT, 25, 1)
    FIELD(CAN1_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(CAN1_REF_CTRL, SRCSEL, 0, 3)
REG32(I2C0_REF_CTRL, 0x140)
    FIELD(I2C0_REF_CTRL, CLKACT, 25, 1)
    FIELD(I2C0_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(I2C0_REF_CTRL, SRCSEL, 0, 3)
REG32(I2C1_REF_CTRL, 0x144)
    FIELD(I2C1_REF_CTRL, CLKACT, 25, 1)
    FIELD(I2C1_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(I2C1_REF_CTRL, SRCSEL, 0, 3)
REG32(DBG_LPD_CTRL, 0x148)
    FIELD(DBG_LPD_CTRL, CLKACT, 25, 1)
    FIELD(DBG_LPD_CTRL, DIVISOR0, 8, 10)
    FIELD(DBG_LPD_CTRL, SRCSEL, 0, 3)
REG32(TIMESTAMP_REF_CTRL, 0x14c)
    FIELD(TIMESTAMP_REF_CTRL, CLKACT, 25, 1)
    FIELD(TIMESTAMP_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(TIMESTAMP_REF_CTRL, SRCSEL, 0, 3)
REG32(CRL_SAFETY_CHK, 0x150)
REG32(PSM_REF_CTRL, 0x154)
    FIELD(PSM_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(PSM_REF_CTRL, SRCSEL, 0, 3)
REG32(DBG_TSTMP_CTRL, 0x158)
    FIELD(DBG_TSTMP_CTRL, CLKACT, 25, 1)
    FIELD(DBG_TSTMP_CTRL, DIVISOR0, 8, 10)
    FIELD(DBG_TSTMP_CTRL, SRCSEL, 0, 3)
REG32(CPM_TOPSW_REF_CTRL, 0x15c)
    FIELD(CPM_TOPSW_REF_CTRL, CLKACT, 25, 1)
    FIELD(CPM_TOPSW_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(CPM_TOPSW_REF_CTRL, SRCSEL, 0, 3)
REG32(USB3_DUAL_REF_CTRL, 0x160)
    FIELD(USB3_DUAL_REF_CTRL, CLKACT, 25, 1)
    FIELD(USB3_DUAL_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(USB3_DUAL_REF_CTRL, SRCSEL, 0, 3)
REG32(RST_CPU_R5, 0x300)
    FIELD(RST_CPU_R5, RESET_PGE, 4, 1)
    FIELD(RST_CPU_R5, RESET_AMBA, 2, 1)
    FIELD(RST_CPU_R5, RESET_CPU1, 1, 1)
    FIELD(RST_CPU_R5, RESET_CPU0, 0, 1)
REG32(RST_ADMA, 0x304)
    FIELD(RST_ADMA, RESET, 0, 1)
REG32(RST_GEM0, 0x308)
    FIELD(RST_GEM0, RESET, 0, 1)
REG32(RST_GEM1, 0x30c)
    FIELD(RST_GEM1, RESET, 0, 1)
REG32(RST_SPARE, 0x310)
    FIELD(RST_SPARE, RESET, 0, 1)
REG32(RST_USB0, 0x314)
    FIELD(RST_USB0, RESET, 0, 1)
REG32(RST_UART0, 0x318)
    FIELD(RST_UART0, RESET, 0, 1)
REG32(RST_UART1, 0x31c)
    FIELD(RST_UART1, RESET, 0, 1)
REG32(RST_SPI0, 0x320)
    FIELD(RST_SPI0, RESET, 0, 1)
REG32(RST_SPI1, 0x324)
    FIELD(RST_SPI1, RESET, 0, 1)
REG32(RST_CAN0, 0x328)
    FIELD(RST_CAN0, RESET, 0, 1)
REG32(RST_CAN1, 0x32c)
    FIELD(RST_CAN1, RESET, 0, 1)
REG32(RST_I2C0, 0x330)
    FIELD(RST_I2C0, RESET, 0, 1)
REG32(RST_I2C1, 0x334)
    FIELD(RST_I2C1, RESET, 0, 1)
REG32(RST_DBG_LPD, 0x338)
    FIELD(RST_DBG_LPD, RPU_DBG1_RESET, 5, 1)
    FIELD(RST_DBG_LPD, RPU_DBG0_RESET, 4, 1)
    FIELD(RST_DBG_LPD, RESET_HSDP, 1, 1)
    FIELD(RST_DBG_LPD, RESET, 0, 1)
REG32(RST_GPIO, 0x33c)
    FIELD(RST_GPIO, RESET, 0, 1)
REG32(RST_TTC, 0x344)
    FIELD(RST_TTC, TTC3_RESET, 3, 1)
    FIELD(RST_TTC, TTC2_RESET, 2, 1)
    FIELD(RST_TTC, TTC1_RESET, 1, 1)
    FIELD(RST_TTC, TTC0_RESET, 0, 1)
REG32(RST_TIMESTAMP, 0x348)
    FIELD(RST_TIMESTAMP, RESET, 0, 1)
REG32(RST_SWDT, 0x34c)
    FIELD(RST_SWDT, RESET, 0, 1)
REG32(RST_OCM, 0x350)
    FIELD(RST_OCM, RESET, 0, 1)
REG32(RST_IPI, 0x354)
    FIELD(RST_IPI, RESET, 0, 1)
REG32(RST_SYSMON, 0x358)
    FIELD(RST_SYSMON, SEQ_RST, 1, 1)
    FIELD(RST_SYSMON, CFG_RST, 0, 1)
REG32(RST_FPD, 0x360)
    FIELD(RST_FPD, SRST, 1, 1)
    FIELD(RST_FPD, POR, 0, 1)
REG32(PSM_RST_MODE, 0x370)
    FIELD(PSM_RST_MODE, WAKEUP, 2, 1)
    FIELD(PSM_RST_MODE, RST_MODE, 0, 2)

#define CRL_R_MAX (R_PSM_RST_MODE + 1)

REG32(VERSAL2_ERR_CTRL, 0x0)
REG32(VERSAL2_WPROT, 0x1c)
    FIELD(VERSAL2_WPROT, ACTIVE, 0, 1)
REG32(VERSAL2_RPLL_CTRL, 0x40)
    FIELD(VERSAL2_RPLL_CTRL, POST_SRC, 24, 3)
    FIELD(VERSAL2_RPLL_CTRL, PRE_SRC, 20, 3)
    FIELD(VERSAL2_RPLL_CTRL, CLKOUTDIV, 16, 2)
    FIELD(VERSAL2_RPLL_CTRL, FBDIV, 8, 8)
    FIELD(VERSAL2_RPLL_CTRL, BYPASS, 3, 1)
    FIELD(VERSAL2_RPLL_CTRL, RESET, 0, 1)
REG32(VERSAL2_RPLL_CFG, 0x44)
    FIELD(VERSAL2_RPLL_CFG, LOCK_DLY, 25, 7)
    FIELD(VERSAL2_RPLL_CFG, LOCK_CNT, 13, 10)
    FIELD(VERSAL2_RPLL_CFG, LFHF, 10, 2)
    FIELD(VERSAL2_RPLL_CFG, CP, 5, 4)
    FIELD(VERSAL2_RPLL_CFG, RES, 0, 4)
REG32(VERSAL2_FLXPLL_CTRL, 0x50)
    FIELD(VERSAL2_FLXPLL_CTRL, POST_SRC, 24, 3)
    FIELD(VERSAL2_FLXPLL_CTRL, PRE_SRC, 20, 3)
    FIELD(VERSAL2_FLXPLL_CTRL, CLKOUTDIV, 16, 2)
    FIELD(VERSAL2_FLXPLL_CTRL, FBDIV, 8, 8)
    FIELD(VERSAL2_FLXPLL_CTRL, BYPASS, 3, 1)
    FIELD(VERSAL2_FLXPLL_CTRL, RESET, 0, 1)
REG32(VERSAL2_FLXPLL_CFG, 0x54)
    FIELD(VERSAL2_FLXPLL_CFG, LOCK_DLY, 25, 7)
    FIELD(VERSAL2_FLXPLL_CFG, LOCK_CNT, 13, 10)
    FIELD(VERSAL2_FLXPLL_CFG, LFHF, 10, 2)
    FIELD(VERSAL2_FLXPLL_CFG, CP, 5, 4)
    FIELD(VERSAL2_FLXPLL_CFG, RES, 0, 4)
REG32(VERSAL2_PLL_STATUS, 0x60)
    FIELD(VERSAL2_PLL_STATUS, FLXPLL_STABLE, 3, 1)
    FIELD(VERSAL2_PLL_STATUS, RPLL_STABLE, 2, 1)
    FIELD(VERSAL2_PLL_STATUS, FLXPLL_LOCK, 1, 1)
    FIELD(VERSAL2_PLL_STATUS, RPLL_LOCK, 0, 1)
REG32(VERSAL2_RPLL_TO_XPD_CTRL, 0x100)
    FIELD(VERSAL2_RPLL_TO_XPD_CTRL, DIVISOR0, 8, 10)
REG32(VERSAL2_LPX_TOP_SWITCH_CTRL, 0x104)
    FIELD(VERSAL2_LPX_TOP_SWITCH_CTRL, CLKACT_ADMA, 26, 1)
    FIELD(VERSAL2_LPX_TOP_SWITCH_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_LPX_TOP_SWITCH_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_LPX_TOP_SWITCH_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_LPX_LSBUS_CLK_CTRL, 0x108)
    FIELD(VERSAL2_LPX_LSBUS_CLK_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_LPX_LSBUS_CLK_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_LPX_LSBUS_CLK_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_RPU_CLK_CTRL, 0x10c)
    FIELD(VERSAL2_RPU_CLK_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_RPU_CLK_CTRL, CLKACT_CLUSTERE, 24, 1)
    FIELD(VERSAL2_RPU_CLK_CTRL, CLKACT_CLUSTERD, 23, 1)
    FIELD(VERSAL2_RPU_CLK_CTRL, CLKACT_CLUSTERC, 22, 1)
    FIELD(VERSAL2_RPU_CLK_CTRL, CLKACT_CLUSTERB, 21, 1)
    FIELD(VERSAL2_RPU_CLK_CTRL, CLKACT_CLUSTERA, 20, 1)
    FIELD(VERSAL2_RPU_CLK_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_RPU_CLK_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_OCM_CLK_CTRL, 0x120)
    FIELD(VERSAL2_OCM_CLK_CTRL, CLKACT_OCM3, 24, 1)
    FIELD(VERSAL2_OCM_CLK_CTRL, CLKACT_OCM2, 23, 1)
    FIELD(VERSAL2_OCM_CLK_CTRL, CLKACT_OCM1, 22, 1)
    FIELD(VERSAL2_OCM_CLK_CTRL, CLKACT_OCM0, 21, 1)
REG32(VERSAL2_IOU_SWITCH_CLK_CTRL, 0x124)
    FIELD(VERSAL2_IOU_SWITCH_CLK_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_IOU_SWITCH_CLK_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_IOU_SWITCH_CLK_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_GEM0_REF_CTRL, 0x128)
    FIELD(VERSAL2_GEM0_REF_CTRL, CLKACT_RX, 27, 1)
    FIELD(VERSAL2_GEM0_REF_CTRL, CLKACT_TX, 26, 1)
    FIELD(VERSAL2_GEM0_REF_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_GEM0_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_GEM0_REF_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_GEM1_REF_CTRL, 0x12c)
    FIELD(VERSAL2_GEM1_REF_CTRL, CLKACT_RX, 27, 1)
    FIELD(VERSAL2_GEM1_REF_CTRL, CLKACT_TX, 26, 1)
    FIELD(VERSAL2_GEM1_REF_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_GEM1_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_GEM1_REF_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_GEM_TSU_REF_CLK_CTRL, 0x130)
    FIELD(VERSAL2_GEM_TSU_REF_CLK_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_GEM_TSU_REF_CLK_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_GEM_TSU_REF_CLK_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_USB0_BUS_REF_CLK_CTRL, 0x134)
    FIELD(VERSAL2_USB0_BUS_REF_CLK_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_USB0_BUS_REF_CLK_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_USB0_BUS_REF_CLK_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_USB1_BUS_REF_CLK_CTRL, 0x138)
    FIELD(VERSAL2_USB1_BUS_REF_CLK_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_USB1_BUS_REF_CLK_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_USB1_BUS_REF_CLK_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_UART0_REF_CLK_CTRL, 0x13c)
    FIELD(VERSAL2_UART0_REF_CLK_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_UART0_REF_CLK_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_UART0_REF_CLK_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_UART1_REF_CLK_CTRL, 0x140)
    FIELD(VERSAL2_UART1_REF_CLK_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_UART1_REF_CLK_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_UART1_REF_CLK_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_SPI0_REF_CLK_CTRL, 0x144)
    FIELD(VERSAL2_SPI0_REF_CLK_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_SPI0_REF_CLK_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_SPI0_REF_CLK_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_SPI1_REF_CLK_CTRL, 0x148)
    FIELD(VERSAL2_SPI1_REF_CLK_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_SPI1_REF_CLK_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_SPI1_REF_CLK_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_CAN0_REF_2X_CTRL, 0x14c)
    FIELD(VERSAL2_CAN0_REF_2X_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_CAN0_REF_2X_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_CAN0_REF_2X_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_CAN1_REF_2X_CTRL, 0x150)
    FIELD(VERSAL2_CAN1_REF_2X_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_CAN1_REF_2X_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_CAN1_REF_2X_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_CAN2_REF_2X_CTRL, 0x154)
    FIELD(VERSAL2_CAN2_REF_2X_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_CAN2_REF_2X_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_CAN2_REF_2X_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_CAN3_REF_2X_CTRL, 0x158)
    FIELD(VERSAL2_CAN3_REF_2X_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_CAN3_REF_2X_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_CAN3_REF_2X_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_I3C0_REF_CTRL, 0x15c)
    FIELD(VERSAL2_I3C0_REF_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_I3C0_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_I3C0_REF_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_I3C1_REF_CTRL, 0x160)
    FIELD(VERSAL2_I3C1_REF_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_I3C1_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_I3C1_REF_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_I3C2_REF_CTRL, 0x164)
    FIELD(VERSAL2_I3C2_REF_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_I3C2_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_I3C2_REF_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_I3C3_REF_CTRL, 0x168)
    FIELD(VERSAL2_I3C3_REF_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_I3C3_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_I3C3_REF_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_I3C4_REF_CTRL, 0x16c)
    FIELD(VERSAL2_I3C4_REF_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_I3C4_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_I3C4_REF_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_I3C5_REF_CTRL, 0x170)
    FIELD(VERSAL2_I3C5_REF_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_I3C5_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_I3C5_REF_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_I3C6_REF_CTRL, 0x174)
    FIELD(VERSAL2_I3C6_REF_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_I3C6_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_I3C6_REF_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_I3C7_REF_CTRL, 0x178)
    FIELD(VERSAL2_I3C7_REF_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_I3C7_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_I3C7_REF_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_DBG_LPX_CTRL, 0x17c)
    FIELD(VERSAL2_DBG_LPX_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_DBG_LPX_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_DBG_LPX_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_TIMESTAMP_REF_CTRL, 0x180)
    FIELD(VERSAL2_TIMESTAMP_REF_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_TIMESTAMP_REF_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_TIMESTAMP_REF_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_SAFETY_CHK, 0x184)
REG32(VERSAL2_ASU_CLK_CTRL, 0x188)
    FIELD(VERSAL2_ASU_CLK_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_ASU_CLK_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_DBG_TSTMP_CLK_CTRL, 0x18c)
    FIELD(VERSAL2_DBG_TSTMP_CLK_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_DBG_TSTMP_CLK_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_DBG_TSTMP_CLK_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_MMI_TOPSW_CLK_CTRL, 0x190)
    FIELD(VERSAL2_MMI_TOPSW_CLK_CTRL, CLKACT, 25, 1)
    FIELD(VERSAL2_MMI_TOPSW_CLK_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_MMI_TOPSW_CLK_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_WWDT_PLL_CLK_CTRL, 0x194)
    FIELD(VERSAL2_WWDT_PLL_CLK_CTRL, DIVISOR0, 8, 10)
    FIELD(VERSAL2_WWDT_PLL_CLK_CTRL, SRCSEL, 0, 3)
REG32(VERSAL2_RCLK_CTRL, 0x1a0)
    FIELD(VERSAL2_RCLK_CTRL, CLKACT, 8, 6)
    FIELD(VERSAL2_RCLK_CTRL, SELECT, 0, 6)
REG32(VERSAL2_RST_RPU_A, 0x310)
    FIELD(VERSAL2_RST_RPU_A, TOPRESET, 16, 1)
    FIELD(VERSAL2_RST_RPU_A, CORE1_POR, 9, 1)
    FIELD(VERSAL2_RST_RPU_A, CORE0_POR, 8, 1)
    FIELD(VERSAL2_RST_RPU_A, CORE1_RESET, 1, 1)
    FIELD(VERSAL2_RST_RPU_A, CORE0_RESET, 0, 1)
REG32(VERSAL2_RST_RPU_B, 0x314)
    FIELD(VERSAL2_RST_RPU_B, TOPRESET, 16, 1)
    FIELD(VERSAL2_RST_RPU_B, CORE1_POR, 9, 1)
    FIELD(VERSAL2_RST_RPU_B, CORE0_POR, 8, 1)
    FIELD(VERSAL2_RST_RPU_B, CORE1_RESET, 1, 1)
    FIELD(VERSAL2_RST_RPU_B, CORE0_RESET, 0, 1)
REG32(VERSAL2_RST_RPU_C, 0x318)
    FIELD(VERSAL2_RST_RPU_C, TOPRESET, 16, 1)
    FIELD(VERSAL2_RST_RPU_C, CORE1_POR, 9, 1)
    FIELD(VERSAL2_RST_RPU_C, CORE0_POR, 8, 1)
    FIELD(VERSAL2_RST_RPU_C, CORE1_RESET, 1, 1)
    FIELD(VERSAL2_RST_RPU_C, CORE0_RESET, 0, 1)
REG32(VERSAL2_RST_RPU_D, 0x31c)
    FIELD(VERSAL2_RST_RPU_D, TOPRESET, 16, 1)
    FIELD(VERSAL2_RST_RPU_D, CORE1_POR, 9, 1)
    FIELD(VERSAL2_RST_RPU_D, CORE0_POR, 8, 1)
    FIELD(VERSAL2_RST_RPU_D, CORE1_RESET, 1, 1)
    FIELD(VERSAL2_RST_RPU_D, CORE0_RESET, 0, 1)
REG32(VERSAL2_RST_RPU_E, 0x320)
    FIELD(VERSAL2_RST_RPU_E, TOPRESET, 16, 1)
    FIELD(VERSAL2_RST_RPU_E, CORE1_POR, 9, 1)
    FIELD(VERSAL2_RST_RPU_E, CORE0_POR, 8, 1)
    FIELD(VERSAL2_RST_RPU_E, CORE1_RESET, 1, 1)
    FIELD(VERSAL2_RST_RPU_E, CORE0_RESET, 0, 1)
REG32(VERSAL2_RST_RPU_GD_0, 0x324)
    FIELD(VERSAL2_RST_RPU_GD_0, RESET, 1, 1)
    FIELD(VERSAL2_RST_RPU_GD_0, TOP_RESET, 0, 1)
REG32(VERSAL2_RST_RPU_GD_1, 0x328)
    FIELD(VERSAL2_RST_RPU_GD_1, RESET, 1, 1)
    FIELD(VERSAL2_RST_RPU_GD_1, TOP_RESET, 0, 1)
REG32(VERSAL2_RST_ASU_GD, 0x32c)
    FIELD(VERSAL2_RST_ASU_GD, RESET, 1, 1)
    FIELD(VERSAL2_RST_ASU_GD, TOP_RESET, 0, 1)
REG32(VERSAL2_RST_ADMA, 0x334)
    FIELD(VERSAL2_RST_ADMA, RESET, 0, 1)
REG32(VERSAL2_RST_SDMA, 0x338)
    FIELD(VERSAL2_RST_SDMA, RESET, 0, 1)
REG32(VERSAL2_RST_GEM0, 0x33c)
    FIELD(VERSAL2_RST_GEM0, RESET, 0, 1)
REG32(VERSAL2_RST_GEM1, 0x340)
    FIELD(VERSAL2_RST_GEM1, RESET, 0, 1)
REG32(VERSAL2_RST_USB0, 0x348)
    FIELD(VERSAL2_RST_USB0, RESET, 0, 1)
REG32(VERSAL2_RST_USB1, 0x34c)
    FIELD(VERSAL2_RST_USB1, RESET, 0, 1)
REG32(VERSAL2_RST_UART0, 0x350)
    FIELD(VERSAL2_RST_UART0, RESET, 0, 1)
REG32(VERSAL2_RST_UART1, 0x354)
    FIELD(VERSAL2_RST_UART1, RESET, 0, 1)
REG32(VERSAL2_RST_SPI0, 0x358)
    FIELD(VERSAL2_RST_SPI0, RESET, 0, 1)
REG32(VERSAL2_RST_SPI1, 0x35c)
    FIELD(VERSAL2_RST_SPI1, RESET, 0, 1)
REG32(VERSAL2_RST_CAN0, 0x360)
    FIELD(VERSAL2_RST_CAN0, RESET, 0, 1)
REG32(VERSAL2_RST_CAN1, 0x364)
    FIELD(VERSAL2_RST_CAN1, RESET, 0, 1)
REG32(VERSAL2_RST_CAN2, 0x368)
    FIELD(VERSAL2_RST_CAN2, RESET, 0, 1)
REG32(VERSAL2_RST_CAN3, 0x36c)
    FIELD(VERSAL2_RST_CAN3, RESET, 0, 1)
REG32(VERSAL2_RST_I3C0, 0x374)
    FIELD(VERSAL2_RST_I3C0, RESET, 0, 1)
REG32(VERSAL2_RST_I3C1, 0x378)
    FIELD(VERSAL2_RST_I3C1, RESET, 0, 1)
REG32(VERSAL2_RST_I3C2, 0x37c)
    FIELD(VERSAL2_RST_I3C2, RESET, 0, 1)
REG32(VERSAL2_RST_I3C3, 0x380)
    FIELD(VERSAL2_RST_I3C3, RESET, 0, 1)
REG32(VERSAL2_RST_I3C4, 0x384)
    FIELD(VERSAL2_RST_I3C4, RESET, 0, 1)
REG32(VERSAL2_RST_I3C5, 0x388)
    FIELD(VERSAL2_RST_I3C5, RESET, 0, 1)
REG32(VERSAL2_RST_I3C6, 0x38c)
    FIELD(VERSAL2_RST_I3C6, RESET, 0, 1)
REG32(VERSAL2_RST_I3C7, 0x390)
    FIELD(VERSAL2_RST_I3C7, RESET, 0, 1)
REG32(VERSAL2_RST_DBG_LPX, 0x398)
    FIELD(VERSAL2_RST_DBG_LPX, RESET_HSDP, 1, 1)
    FIELD(VERSAL2_RST_DBG_LPX, RESET, 0, 1)
REG32(VERSAL2_RST_GPIO, 0x39c)
    FIELD(VERSAL2_RST_GPIO, RESET, 0, 1)
REG32(VERSAL2_RST_TTC, 0x3a0)
    FIELD(VERSAL2_RST_TTC, TTC7_RESET, 7, 1)
    FIELD(VERSAL2_RST_TTC, TTC6_RESET, 6, 1)
    FIELD(VERSAL2_RST_TTC, TTC5_RESET, 5, 1)
    FIELD(VERSAL2_RST_TTC, TTC4_RESET, 4, 1)
    FIELD(VERSAL2_RST_TTC, TTC3_RESET, 3, 1)
    FIELD(VERSAL2_RST_TTC, TTC2_RESET, 2, 1)
    FIELD(VERSAL2_RST_TTC, TTC1_RESET, 1, 1)
    FIELD(VERSAL2_RST_TTC, TTC0_RESET, 0, 1)
REG32(VERSAL2_RST_TIMESTAMP, 0x3a4)
    FIELD(VERSAL2_RST_TIMESTAMP, RESET, 0, 1)
REG32(VERSAL2_RST_SWDT0, 0x3a8)
    FIELD(VERSAL2_RST_SWDT0, RESET, 0, 1)
REG32(VERSAL2_RST_SWDT1, 0x3ac)
    FIELD(VERSAL2_RST_SWDT1, RESET, 0, 1)
REG32(VERSAL2_RST_SWDT2, 0x3b0)
    FIELD(VERSAL2_RST_SWDT2, RESET, 0, 1)
REG32(VERSAL2_RST_SWDT3, 0x3b4)
    FIELD(VERSAL2_RST_SWDT3, RESET, 0, 1)
REG32(VERSAL2_RST_SWDT4, 0x3b8)
    FIELD(VERSAL2_RST_SWDT4, RESET, 0, 1)
REG32(VERSAL2_RST_IPI, 0x3bc)
    FIELD(VERSAL2_RST_IPI, RESET, 0, 1)
REG32(VERSAL2_RST_SYSMON, 0x3c0)
    FIELD(VERSAL2_RST_SYSMON, CFG_RST, 0, 1)
REG32(VERSAL2_ASU_MB_RST_MODE, 0x3c4)
    FIELD(VERSAL2_ASU_MB_RST_MODE, WAKEUP, 2, 1)
    FIELD(VERSAL2_ASU_MB_RST_MODE, RST_MODE, 0, 2)
REG32(VERSAL2_FPX_TOPSW_MUX_CTRL, 0x3c8)
    FIELD(VERSAL2_FPX_TOPSW_MUX_CTRL, SELECT, 0, 1)
REG32(VERSAL2_RST_FPX, 0x3d0)
    FIELD(VERSAL2_RST_FPX, SRST, 1, 1)
    FIELD(VERSAL2_RST_FPX, POR, 0, 1)
REG32(VERSAL2_RST_MMI, 0x3d4)
    FIELD(VERSAL2_RST_MMI, POR, 0, 1)
REG32(VERSAL2_RST_OCM, 0x3d8)
    FIELD(VERSAL2_RST_OCM, RESET_OCM3, 3, 1)
    FIELD(VERSAL2_RST_OCM, RESET_OCM2, 2, 1)
    FIELD(VERSAL2_RST_OCM, RESET_OCM1, 1, 1)
    FIELD(VERSAL2_RST_OCM, RESET_OCM0, 0, 1)

#define VERSAL2_CRL_R_MAX (R_VERSAL2_RST_OCM + 1)

struct XlnxVersalCRLBase {
    SysBusDevice parent_obj;

    uint32_t *regs;
};

struct XlnxVersalCRLBaseClass {
    SysBusDeviceClass parent_class;

    DeviceState ** (*decode_periph_rst)(XlnxVersalCRLBase *s, hwaddr, size_t *);
};

struct XlnxVersalCRL {
    XlnxVersalCRLBase parent_obj;
    qemu_irq irq;

    struct {
        DeviceState *rpu[2];
        DeviceState *adma[8];
        DeviceState *uart[2];
        DeviceState *gem[2];
        DeviceState *usb[1];
    } cfg;

    uint32_t regs[CRL_R_MAX];
    RegisterInfo regs_info[CRL_R_MAX];
};

struct XlnxVersal2CRL {
    XlnxVersalCRLBase parent_obj;

    struct {
        DeviceState *rpu[10];
        DeviceState *adma[8];
        DeviceState *sdma[8];
        DeviceState *uart[2];
        DeviceState *gem[2];
        DeviceState *usb[2];
        DeviceState *can[4];
    } cfg;

    RegisterInfo regs_info[VERSAL2_CRL_R_MAX];
    uint32_t regs[VERSAL2_CRL_R_MAX];
};

static inline const char *xlnx_versal_crl_class_name(VersalVersion ver)
{
    switch (ver) {
    case VERSAL_VER_VERSAL:
        return TYPE_XLNX_VERSAL_CRL;
    case VERSAL_VER_VERSAL2:
        return TYPE_XLNX_VERSAL2_CRL;
    default:
        g_assert_not_reached();
    }
}

#endif
